package ch.frankel.blog.hibernate;

import static org.testng.Assert.assertEquals;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;

import org.hibernate.PropertyValueException;
import org.hibernate.Session;
import org.testng.annotations.Test;

import ch.frankel.blog.hibernate.model.Account;
import ch.frankel.blog.hibernate.model.Customer;
import ch.frankel.blog.hibernate.model.Order;

/**
 * Test class to check some basic truths about Hibernate.
 * 
 * @author Nicolas Frnkel
 * @since 14 juin 2009
 */
public class TestPart2 extends AbstractTest {

    /**
     * Test class for checking whether you need to add both end to an
     * association.
     * 
     * @throws SQLException
     */
    @Test(expectedExceptions = PropertyValueException.class)
    public void testAssociationCustomerEnd() throws SQLException {

        Session session = factory.getCurrentSession();

        session.getTransaction().begin();

        try {

            Customer customer = (Customer) session.load(Customer.class, 1L);

            Order order = new Order();

            order.setOrderDate(new Date());

            customer.getOrders().add(order);

        } finally {

            session.getTransaction().commit();
        }
    }

    /**
     * Test class for checking whether you need to add both end to an
     * association.
     * 
     * @throws SQLException
     */
    @Test
    public void testAssociationBothEnds() throws SQLException {

        Session session = factory.openSession();

        session.getTransaction().begin();

        Customer customer;

        int size;

        try {

            customer = (Customer) session.load(Customer.class, 1L);

            Order order = new Order();

            order.setOrderDate(new Date());

            customer.addOrder(order);

            size = customer.getOrders().size();

        } finally {

            session.getTransaction().commit();
        }

        long id = customer.getId();

        Connection connection = DriverManager.getConnection("jdbc:derby:target/myDb");

        try {

            PreparedStatement stmt = connection.prepareStatement("SELECT COUNT(*) FROM ORDERS WHERE CUSTOMER_ID = ?");

            stmt.setLong(1, id);

            ResultSet rs = stmt.executeQuery();

            rs.next();

            int count = rs.getInt(1);

            assertEquals(count, size);

        } finally {

            connection.close();
        }
    }

    /**
     * Test class for checking whether you need to remove both end to an
     * association.
     * 
     * @throws SQLException
     */
    @Test
    public void testRemoveAssociation() throws SQLException {

        Session session = factory.openSession();

        session.getTransaction().begin();

        Order order = (Order) session.load(Order.class, 1L);

        Customer customer = order.getCustomer();

        session.delete(order);

        customer.getOrders().remove(order);

        session.getTransaction().commit();

        int size = customer.getOrders().size();

        long id = customer.getId();

        Connection connection = DriverManager.getConnection("jdbc:derby:target/myDb");

        PreparedStatement stmt = connection.prepareStatement("SELECT COUNT(*) FROM ORDERS WHERE CUSTOMER_ID = ?");

        stmt.setLong(1, id);

        ResultSet rs = stmt.executeQuery();

        rs.next();

        int count = rs.getInt(1);

        connection.close();

        assertEquals(count, size);
    }

    @Test
    public void testReassociation() throws SQLException {

        Session session = factory.openSession();

        session.getTransaction().begin();

        Customer customer = (Customer) session.load(Customer.class, 2L);

        Account account = (Account) session.load(Account.class, 3L);

        // Remove
        for (Customer oldCustomer : account.getCustomers()) {

            oldCustomer.removeAccount(account);
        }

        // Add
        customer.addAccount(account);

        int size = customer.getAccounts().size();

        session.getTransaction().commit();

        Connection connection = DriverManager.getConnection("jdbc:derby:target/myDb");

        PreparedStatement stmt = connection
                .prepareStatement("SELECT COUNT(*) FROM ACCOUNT a JOIN ACCOUNT_CUSTOMER ac ON a.ID = ac.ACCOUNT_ID WHERE ac.CUSTOMER_ID = ?");

        stmt.setLong(1, customer.getId());

        ResultSet rs = stmt.executeQuery();

        rs.next();

        int count = rs.getInt(1);

        connection.close();

        assertEquals(count, size);
    }
}
